function [x, nit] = nlnewtonP(nlfc, J, x0, itfc)
% Find zero of a nonlinear function by the Newton's method.
%
% Usage:   x        = nlnewton(nlfc, J, x0)
%        [ x, nit ] = nlnewton(nlfc, J, x0)
% INPUT:
% nlfc     - handle to the nonlinear function, f(x);
% J        - handle to the function computing functions Jacobian matrix, A = J(x);
% x0       - initial guess vector;
% (optional:)
% itfc     - an auxiliary function executed at every iteration step,
% OUTPUT:
% x        - n-by-1 vector solving f(x)=0; 
% (optional:)
% nit      - number of iterations performed until convergence;
%
% Examples: 
% 1D:
%  f=@(x) x.^2-2; J=@(x) 2*x; x0=1; x=nlnewton(f,J,x0);
% 2D:
%  f = @(x) [x(1)+sin(x(2)); x(1)*cos(x(2))+x(2)]; 
%  J = @(x) [1 cos(x(2)); cos(x(2)) -x(1)*sin(x(2))+1 ];
%  x0 = [1 1]';
%  x = nlnewton(f,J, x0);
%
% See also: nlsecant, nlbreuden.

% Input check:
% error(nargchk(3,4,nargin));

% Defaults:
maxIt = 100;     % maximum number of iterations
th  = 1e-14;     % threshold
nit = 0;         % iterations counter
if nargin<4, itfc = []; end;

% Start iterations:
x = x0;
dx = [0 0]';
if ~isempty(itfc), itfc(nit,x,dx,nlfc); end
while abs(nlfc(x)) > th
    nit = nit + 1;
    if nit > maxIt
        warning(['Convergence failed: maximum number of iterations (' num2str(maxIt) ') was exceeded.']);
        return;
    end
    
    dx = -nlfc(x)/J(x);
    if ~isempty(itfc), itfc(nit,x,dx,nlfc); end
    x = x + dx;
    
end

if nargout < 2
    display( [ '# of iterations: ' num2str(nit)] )
end